from pyparsing import Literal, Word, Optional, printables, nums

from codeable_detectors.basic_detectors import AtLeastOneFileMatchesDetector, MatchesAcrossMultipleFilesDetector, FunctionInvocation
from codeable_detectors.evidences import FailedEvidence, Evidence, ServiceEvidence, Evidences


class PHPAccessToServerRequestMethod(AtLeastOneFileMatchesDetector):
    def __init__(self):
        super().__init__()
        self.file_endings = ["php"]

    def detect_in_context(self, ctx, **kwargs):
        line_matches = ctx.matches_pattern(Literal("$") + Literal("_SERVER") + Literal("[") +
                                           Word("'\"", max=1) + Literal("REQUEST_METHOD") + Word("'\"", max=1) +
                                           Literal("]"))
        if not line_matches:
            FailedEvidence("PHP access to server request method not found")
        return Evidence(line_matches)


class PHPRestfulService(MatchesAcrossMultipleFilesDetector):
    def __init__(self):
        super().__init__([PHPAccessToServerRequestMethod(),
                          # match something like: header('Content-Type: application/json');
                          FunctionInvocation("header", "php", Word("'\"", max=1) + Literal("Content-Type") +
                                             Literal(":") + Literal("application/json") + Word("'\"", max=1)),
                          # match something like: header('HTTP/1.1[...]');
                          FunctionInvocation("header", "php", Word("'\"", max=1) +
                                             Literal("HTTP/") + Word(nums, max=1) + Literal(".") + Word(nums, max=1) +
                                             Optional(Word(printables, excludeChars="'\"")) + Word("'\"", max=1))])

    def detect(self, directory, **kwargs):
        evidences = super().detect(directory)
        if evidences.have_failed():
            return evidences
        elif evidences.is_empty():
            return FailedEvidence("no evidences for all codeable_detectors of php restful services found")
        return Evidences(ServiceEvidence(evidences.get_all_matches()).set_properties(
            detector_technology_types=["php", "restful"], kwargs=kwargs))
